/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc.groups;

import rdc.RDCSample;

import com.inet.report.Area;
import com.inet.report.BorderProperties;
import com.inet.report.DatabaseField;
import com.inet.report.DatabaseTables;
import com.inet.report.Datasource;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.Group;
import com.inet.report.RDC;
import com.inet.report.Section;
import com.inet.report.SpecialField;
import com.inet.report.SummaryField;
import com.inet.report.TableSource;
import com.inet.report.Text;
import com.inet.report.TextPart;
import com.inet.report.ValueProperties;

/**
 * This sample shows how to build a report with groups. The data will be sorted by customer name and by order id. This
 * shows order amount for each order and sum of order amount for each customer. On the last page this shows sum of
 * amount for all customers.<BR>
 * <BR>
 * The Data Source Configuration "Sample Database" is necessary for this sample. You can create it using the Data Source
 * Manager in i-net Designer or Configuration Manager.
 */
public class GroupSample extends RDCSample {
    static final String[] CUSTOMERS_COLUMN_NAMES = { "CustomerID", "ContactName", "ContactTitle", "ContactPosition", "LastYearsSales", "Address",
        "Address2", "City", "Region", "Country", "PostalCode", "Phone", "Fax" };
    static final int[]    CUSTOMERS_COLUMN_TYPES = { Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.NUMBER, Field.STRING,
        Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.STRING };
    static final String[] ORDERS_COLUMN_NAMES    = { "OrderID", "OrderAmount", "CustomerID", "EmployeeID", "OrderDate", "RequiredDate", "ShipDate",
        "ShipVia", "Shipped", "PO#", "PaymentReceived" };
    static final int[]    ORDERS_COLUMN_TYPES    = { Field.NUMBER, Field.NUMBER, Field.STRING, Field.NUMBER, Field.DATETIME, Field.DATETIME,
        Field.DATETIME, Field.STRING, Field.BOOLEAN, Field.STRING, Field.BOOLEAN };

    /**
     * {@inheritDoc}
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // Create a new Engine
            Engine eng = RDC.createEmptyEngine( exportFmt );

            // Set database connection
            final DatabaseTables dbTables = eng.getDatabaseTables();
            final Fields fields = eng.getFields();

            // Set data source properties
            Datasource ds = dbTables.createDatasource( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Create Tables
            TableSource ts_Customers = ds.createTableSource( "Customers" );
            for( int colIdx = 0; colIdx < CUSTOMERS_COLUMN_NAMES.length; colIdx++ ) {
                ts_Customers.addColumn( CUSTOMERS_COLUMN_NAMES[colIdx], CUSTOMERS_COLUMN_TYPES[colIdx] );
            }

            TableSource ts_Orders = ds.createTableSource( "Orders" );
            for( int colIdx = 0; colIdx < ORDERS_COLUMN_NAMES.length; colIdx++ ) {
                ts_Orders.addColumn( ORDERS_COLUMN_NAMES[colIdx], ORDERS_COLUMN_TYPES[colIdx] );
            }

            // Link the tables
            dbTables.addJoin( ts_Customers.getAlias(), "CustomerID", ts_Orders.getAlias(), "CustomerID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );

            // Add database fields to report
            DatabaseField dbF_customerName = fields.getDatabaseField( ts_Customers.getAlias() + ".ContactName" );
            DatabaseField dbF_orderAmount = fields.getDatabaseField( ts_Orders.getAlias() + ".OrderAmount" );
            DatabaseField dbF_orderDate = fields.getDatabaseField( ts_Orders.getAlias() + ".OrderDate" );
            DatabaseField dbF_paymentReceived = fields.getDatabaseField( ts_Orders.getAlias() + ".PaymentReceived" );
            DatabaseField dbF_orderId = fields.getDatabaseField( ts_Orders.getAlias() + ".OrderID" );
            //API-Change: all special fields were added at report creation time, use get instead of add
            SpecialField printDate = fields.getSpecialField( SpecialField.PRINT_DATE );
            SpecialField pageCount = fields.getSpecialField( SpecialField.PAGE_NUMBER );

            // Add subscriptions to page header section
            Area phArea = eng.getArea( "PH" );
            Section phSec = phArea.getSection( 0 );
            phSec.setBackColor( 0x00FF0000 );
            Text customerNameText = phSec.addText( 60, 690, 5425, 230 );
            TextPart customerNameTextPart = customerNameText.addParagraph().addTextPart( "CustomerName" );
            customerNameText.setCanGrow( false );
            customerNameTextPart.setBold( true );
            customerNameTextPart.setUnderline( true );
            Text orderIDText = phSec.addText( 5605, 690, 1043, 230 );
            TextPart orderIDTextPart = orderIDText.addParagraph().addTextPart( "OrderID" );
            orderIDText.setCanGrow( false );
            orderIDTextPart.setBold( true );
            orderIDTextPart.setUnderline( true );
            Text orderAmountText = phSec.addText( 6768, 690, 1043, 230 );
            TextPart orderAmountTextPart = orderAmountText.addParagraph().addTextPart( "OrderAmount" );
            orderAmountText.setCanGrow( false );
            orderAmountTextPart.setBold( true );
            orderAmountTextPart.setUnderline( true );
            Text orderDateText = phSec.addText( 7931, 690, 1877, 230 );
            TextPart orderDateTextPart = orderDateText.addParagraph().addTextPart( "OrderDate" );
            orderDateText.setCanGrow( false );
            orderDateTextPart.setBold( true );
            orderDateTextPart.setUnderline( true );
            Text paymentText = phSec.addText( 9928, 690, 1039, 230 );
            TextPart paymentTextPart = paymentText.addParagraph().addTextPart( "PaymentReceived" );
            paymentText.setCanGrow( false );
            paymentTextPart.setBold( true );
            paymentTextPart.setUnderline( true );

            // Add 'print date' to page header section
            FieldElement printDateFE = phSec.addFieldElement( printDate, 60, 230, 1468, 230 );
            printDateFE.setDateFormatType( ValueProperties.USE_SYSTEM_SHORT_DATE );

            // Add 'page count' to page footer section
            Area pfArea = eng.getArea( "PF" );
            Section pfSec = pfArea.getSection( 0 );
            pfSec.addFieldElement( pageCount, 9131, 460, 1836, 230 );

            // Add database fields to detail section
            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );
            dSec.addFieldElement( dbF_customerName, 60, 0, 5425, 230 );
            dSec.addFieldElement( dbF_orderId, 5605, 0, 1043, 230 );
            dSec.addFieldElement( dbF_orderAmount, 6768, 0, 1043, 230 );
            FieldElement od = dSec.addFieldElement( dbF_orderDate, 7931, 0, 1877, 230 );
            od.setDateFormatType( ValueProperties.USE_SYSTEM_SHORT_DATE );
            dSec.addFieldElement( dbF_paymentReceived, 9928, 0, 1039, 230 );

            // Add a group
            Group group1 = eng.addGroup( dbF_customerName );

            // Add a groupname field to group header
            Area gh1Area = group1.getHeader();
            Section gh1Sec = gh1Area.getSection( 0 );
            FieldElement gNField1a = gh1Sec.addFieldElement( fields.getGroupNameField( 0 ), 60, 230, 5425, 230 );
            gNField1a.setBold( true );
            gNField1a.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            // Add some summary fields to group footer
            Area gf1Area = group1.getFooter();
            Section gf1Sec = gf1Area.getSection( 0 );
            FieldElement gNField1b = gf1Sec.addFieldElement( fields.getGroupNameField( 0 ), 60, 270, 5425, 230 );
            gNField1b.setBold( true );
            SummaryField smFieldGr1 = fields.addSummaryField( dbF_orderAmount, SummaryField.SUM, "Sum of " + dbF_orderAmount.getName() + "1" );
            smFieldGr1.setGroup( group1 );
            FieldElement ordersAmountSum = gf1Sec.addFieldElement( smFieldGr1, 6768, 270, 1043, 240 );
            ordersAmountSum.setBold( true );
            ordersAmountSum.setTopLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            ordersAmountSum.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            // Add a second group
            Group group2 = eng.addGroup( dbF_orderId );

            // Add a groupname field to group header
            Area gh2Area = group2.getHeader();
            Section gh2Sec = gh2Area.getSection( 0 );
            FieldElement gNField2a = gh2Sec.addFieldElement( fields.getGroupNameField( 1 ), 5605, 230, 1043, 230 );
            gNField2a.setBold( true );
            gNField2a.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            // Add some summmary fields to group footer
            Area gf2Area = group2.getFooter();
            Section gf2Sec = gf2Area.getSection( 0 );
            SummaryField smFieldGr2 = fields.addSummaryField( dbF_orderAmount, SummaryField.SUM, "Sum of " + dbF_orderAmount.getName() + "2" );
            smFieldGr2.setGroup( group2 );
            FieldElement gNField2b = gf2Sec.addFieldElement( fields.getGroupNameField( 1 ), 5605, 270, 1043, 230 );
            gNField2b.setBold( true );
            FieldElement amountOrdersAmount = gf2Sec.addFieldElement( smFieldGr2, 6768, 270, 1043, 240 );
            amountOrdersAmount.setBold( true );
            amountOrdersAmount.setTopLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            // Add total amount to report footer
            Area rfArea = eng.getArea( "RF" );
            Section rfSec = rfArea.getSection( 0 );
            Text totalAmountText = rfSec.addText( 60, 0, 5425, 230 );
            TextPart totalAmountTextPart = totalAmountText.addParagraph().addTextPart( "Total:" );
            totalAmountTextPart.setBold( true );
            totalAmountText.setBottomLineStyle( BorderProperties.LINE_STYLE_DOUBLE );
            SummaryField smFieldTotal = fields.addSummaryField( dbF_orderAmount, SummaryField.SUM, "Sum of " + dbF_orderAmount.getName() + "3" );
            FieldElement totalAmount = rfSec.addFieldElement( smFieldTotal, 6768, 0, 1043, 240 );
            totalAmount.setBold( true );
            totalAmount.setBottomLineStyle( BorderProperties.LINE_STYLE_DOUBLE );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of the sample.
     * @param args no parameters used
     */
    public static void main( String[] args ) {
        new GroupSample().initUI();
    }
}
